#!/usr/bin/env perl

# Enable Perl warnings
use strict;
use warnings;

# Load libraries
use threads;
use Term::ANSIColor;	# Module for colors and errors
use English;
use LWP;
use XML::Simple;
use XML::XPath;
use File::Basename;
use Getopt::Long qw(:config no_ignore_case bundling);
use Data::Dumper;
use Term::ReadKey;
use MySQL::Backup;

# Load internal libraries
use blast;
use datab;
use ipr_scan;

# Print usage and exit if requested - NAPOVEDA; Kontrola poctu parametru.
if (scalar(@ARGV) == 0) 
{
	print STDERR color("bold red"), "ERROR: Bad number of arguments.\n\n", color("reset");
	&print_usage();
	exit(0);
}

# GLOBAL VARIABLESS
# Default parameters values (if out pref is - results has been written to the STD out)
our %params  = (
	'align'			=> 20,	
	'blast_mode'		=> 'blastn',
	'cut_hist'		=> 0.5,	
	'db_blast'    		=> 'nt',	
	'db_heslo'		=> undef,
	'db_host'		=> 'localhost',
	'db_name'		=> undef,
	'db_user'		=> undef,
	'email'			=> '',
	'e_value'		=> 0.001,
	'help'			=> 0,
	'input'			=> '',
	'local'			=> 0,
	'log_cut'		=> 2.0,
	'max_count_hit' 	=> 20,
	'max_threads'		=> 2,
	'mode' 			=> 'analysis',
	'pAdj_cut'		=> 0.01,
	'out_pref'		=> 'out',
	'out_format'		=> 'txt',
	'seq_id'		=> undef,
	'sequence'		=> undef
);

###################################################################################################
########################################### OPTIONS ###############################################
###################################################################################################

# Extract parameters values
GetOptions(
	'align_length=n'		=> \$params{'align'},			# Minimum length of align sequence
	'blast_mode=s'			=> \$params{'blast_mode'},		# Blast mode - blastn,blastp,blastx,tblastn,tblastx
	'cut_hist=s'			=> \$params{'cut_hist'},			# Cutoff value for histogram STEP
	'db_blast=s'			=> \$params{'db_blast'},			# Database for BLAST - nt default
	'db_host=s'			=> \$params{'db_host'},			# Host for DB system - default = localhost
	'db_name=s'			=> \$params{'db_name'},			# Name of database
	'db_user=s'			=> \$params{'db_user'},			# Name of user
	'email=s'			=> \$params{'email'},			# User e-mail address
	'e_value=s'			=> \$params{'e_value'},			# E-value cut off - default 1E-03
	'help'				=> \$params{'help'},				# If we want help message
	'input=s'			=> \$params{'input'},			# Input file
	'local'				=> \$params{'local'},			# If LOCAL BLAST
	'log2f_cut=s'			=> \$params{'log_cut'},			# Cutoff value for LOG2FOLD
	'max_number_hit=n'		=> \$params{'max_count_hit'},	# Maximum number of hits
	'threads=n'			=> \$params{'max_threads'},		# Max count of threads	
	'mode=s'			=> \$params{'mode'},				# User defined mode	
	'pAdj_cut=s'			=> \$params{'pAdj_cut'},			# pAdj cutoff value - default 0.01
	'out=s'				=> \$params{'out_pref'},			# Output prefix for files - default out
	'out_format=s'			=> \$params{'out_format'}		# Output format type - default txt
);

###################################################################################################
###################################### VERIFY PARAMETERS ##########################################
###################################################################################################

# Print help if required
if ($params{'help'}) 
{
	&print_usage();
	exit(0);
}

# Verify parameter MODE
if ($params{'mode'} eq 'create' || $params{'mode'} eq 'repair' || $params{'mode'} eq 'update' ||
	$params{'mode'} eq 'analysis' || $params{'mode'} eq 'delete' || $params{'mode'} eq 'show' ||
	$params{'mode'} eq 'import' || $params{'mode'} eq 'export'
)
{
	print STDERR color("green"), "MESSAGE: Started apllication in mode ", color ("bold green"), $params{'mode'}, color("reset"), color("green"), ".\n", color ("reset");
	&datab::warr_message("############################################################\n");
	&datab::warr_message("INFO: Started apllication in mode " . $params{'mode'} . "\n");
	&datab::warr_message("############################################################\n");
		
	# ERROR LOGFILE
	&datab::err_message("############################################################\n");
	&datab::err_message("INFO: Started aplication in mode " . $params{'mode'} . "\n");
	&datab::err_message("############################################################\n");
}
else 
{
	print STDERR color("bold red"), "ERROR: Unknown mode. Please check the manual.\n\n", color("reset");
	&print_usage();
	exit(0);
}

###################################################################################################
################################# PASSWORD AND GLOBAL VARIABLES ###################################
###################################################################################################

# Get password
print "Password: ";
ReadMode 'noecho';
$params{'db_heslo'} = <STDIN>; 
ReadMode 'original';
$params{'db_heslo'} =~ s/\n//g;
print "\n";

# Global variables for program
our $db = undef;		# Database
our @vlakna_b;			# Field for BLAST threads
our @vlakna_ipr;		# Field for IPR threads
our $work_thread = 1;		# Work thread index
our $thread_control = 1;	# Control wariable for create thread
our $repair_mode = 0;		# Counter for repair mode
our $vlakno_GO = undef;		# Thread for download GO data
our $n_err = 0;			# Number of SEQ with status < 2 or ALL SEQ in DB
our $id = 0;			# Counter of SEQ

# Control MAX_THREADS
if ($params{'max_threads'} > 14)
{
	$params{'max_threads'} = 14;
}

###################################################################################################
################################# MAIN PART OF PROGRAM ############################################
###################################################################################################

########################################################
#1# = If program mode is create
########################################################

if (	defined($params{'email'}) && defined($params{'input'}) && defined($params{'db_name'}) && 
		defined($params{'db_user'}) && 	defined($params{'db_heslo'}) && $params{'mode'} eq 'create'
  )
{
	#1 Create and check database
	if (!(&creat_check_db())) 
	{
		exit (0);
	}

	#2 If INPUT FILE not exist only create database and download data from GO
	if ($params{'input'} eq '') 
	{
		#A Print warrning
		print STDERR color ("red"), "WARNING: Input file is not avaible. Only database MySQL will be created.\n", color("reset");
		&datab::warr_message("WARNING: Input file is not avaible. Only database MySQL will be created.\n");

		#B Download data from database gene ontology
		if (!(&datab::dwl_structure(%params))) 
		{
			exit (0);
		}
		
		#C Exit if success
		exit (1);
	}

	#3 If INPUT FILE is provided and EMAIL is not provided
	if ($params{'email'} eq '')
	{
		#A Print warrning
        	print STDERR color ("red"), "WARNING: Email must be provided for searching. Please use update mode with input file and correct email adresss for searching.\n", color("reset");
        	&datab::warr_message("WARNING: Email must be provided for searching. Please use update mode with input file and correct email adresss for searching.\n");
		
		#B Download data from database gene ontology
		if (!(&datab::dwl_structure(%params))) 
		{
			exit (0);
		}		
		
		#C Exit
		exit (3);		
	}

	# IF INPUT FILE AND EMAIL ARE PROVIDED
	#4 Create thread for download data from GENE ONTOLOGY
	$vlakno_GO = threads->new(\&datab::dwl_structure, %params);
	if (!(defined($vlakno_GO))) 
	{
		print STDERR color("red"), "ERROR: Problem with creating GO thread. Please check manual.\n" ,color("reset");
		&datab::err_message("ERROR: Problem with creating GO thread. Please check manual.\n");
		exit (0);
	}

	#5 OPEN INPUT FILE - if unsuccess
	if (!(open (ZDROJ, $params{'input'}))) 
	{
		#A Print warning
		print STDERR color("red"), "ERROR: Input file cannot be open.\n", color("reset");
		&datab::err_message("ERROR: Input file cannot be open.\n");
		print STDERR color("blue"), "MESSAGE: Program waiting for all data from GO database.\n", color("reset");
		
		#B Waiting on data from GO DATABAZE
		while (!($vlakno_GO->is_joinable())) 
		{
			sleep 2;
		}
		$vlakno_GO->join;
		$vlakno_GO = undef;
		exit (0);
	}

	#6 Get acces to DATABASE
	$db = undef;
	$db = &datab::pristup($params{'db_host'}, $params{'db_name'}, $params{'db_user'}, $params{'db_heslo'});
	if (!(defined($db))) 
	{
		exit (0);	# If problem then exit
	}

	#7 WORK WITH INPUT FILE - FUNCTION
    	&work_with_infile(*ZDROJ);
    	close(ZDROJ);

	#8 REPAIR MODE - count of errors
	$n_err = 0;
	$n_err = $db->do("SELECT name FROM SEKVENCE WHERE status < 2");
	
	#9 Print message for first repair mode
	if ($n_err > 0)
	{
		print STDERR color("blue"), "MESSAGE: Searching completed. Program will be launch in REPAIR mode.\n", color("reset");
		&datab::warr_message("MESSAGE: Searching completed. Program will be launch in REPAIR mode.\n");
	}
	
	#10 REPAIR LOOP
	while (($n_err > 0) && ($repair_mode < 6)) 
	{
		#A Increase repair mode counter
		$repair_mode++;
		#B Print MESSAGE for repair
		print STDERR color("blue"), "MESSAGE: In database was finding $n_err errors for repair in repair cycle $repair_mode.\n", color("reset");
		&datab::warr_message("MESSAGE: In program was finding $n_err errors for repair in repair cycle $repair_mode.\n");
		print STDERR color("blue"), "MESSAGE: Program will start with repair errors in search of sequencess.\n", color("reset");
		&datab::warr_message("MESSAGE: Program will start with repair errors in search of sequencess.\n");
		#C PROVIDE REPAIR
		&update_repair_func($db);
		#D Detection number of errors
		$n_err = 0;
		$n_err = $db->do("SELECT 1_seq_name FROM 1_Sequence WHERE 1_status < 2");
	}

	#11 Waiting if download from GO DATABASE is still running
	if (defined($vlakno_GO)) 
	{
		#A Print warning message
		print STDERR color("blue"), "MESSAGE: Program waiting for download all data from gene ontology database.\n", color("reset");
		&datab::warr_message("MESSAGE: Program waiting for download all data from gene ontology database.\n");
		#B Waiting on GO by loop
		while ((!($vlakno_GO->is_joinable())) && defined($vlakno_GO)) 
		{
			sleep 5;
		}
		#C If download is done finish it
		if (defined($vlakno_GO)) 
		{
			$vlakno_GO->join;
			$vlakno_GO = undef;
		}	
	}
	
	#12 Print message about annotation able sequences
	$n_err = 0;
	$n_err = $db->do("SELECT 1_seq_name FROM 1_Sequence WHERE 1_status < 2");
	if ($n_err > 0)
	{
		print STDERR color("blue"), "MESSAGE: In database can be still repair $n_err of sequences. See the manual for repair mode.\n", color("reset");
		&datab::warr_message("MESSAGE: In database can be still repair $n_err of sequences. See the manual for repair mode.\n");
	}
	else
	{
		print STDERR color("blue"), "MESSAGE: All possible sequences were annotated.\n", color("reset");
		&datab::warr_message("MESSAGE: All possible sequences were annotated.\n");
	}

	#13 Compute GO dependicies
	if (!(&datab::calculating_GOtree($db))) 
	{
		$db->disconnect;
		exit (0);
	}

	#14 Export stat file
	&datab::stat_file($params{'db_host'}, $params{'db_name'}, $params{'db_user'}, $params{'db_heslo'}, 0, 0, 0);
	
	#15 Mode create succesfully finished
	print STDERR color("green"), "MESSAGE: Program launch is completed in CREATE mode.\n", color("reset");
	&datab::warr_message("MESSAGE: Program launch is completed in CREATE mode.\n");
	$db->disconnect;
	exit (1);
}

########################################################
#2# = If program mode is update or repair
########################################################

if (	defined($params{'email'}) && defined($params{'db_name'}) && defined($params{'db_user'}) && 
		defined($params{'db_heslo'}) && (($params{'mode'} eq 'update') || ($params{'mode'} eq 'repair'))
  )
{	
	#1 Database conection
	$db = &datab::pristup($params{'db_host'}, $params{'db_name'}, $params{'db_user'}, $params{'db_heslo'});
	if (!(defined($db))) 
	{
		exit (0);	# Problem with database connection
	}

	#2 Verify structure of database -> if incorrect program end
	if (!(&datab::verify_datab_structure($db,0))) 
	{
		$db->disconnect;
		exit (0);
	}

	#3 Set status = 0 for error sequences where status = 1
	$db->do('UPDATE 1_Sequence SET 1_status = 0 WHERE 1_status = 1;');	
	
	#4A REPAIR OR UPDATE WITHOUT FILE
	if ($params{'input'} eq '') 
	{
		#A Get numbers of records in DB
		$n_err = 0;
		$n_err = $db->do("SELECT * FROM 1_Sequence;");
		print STDERR color("blue"), "MESSAGE: In database is currently storage $n_err of records.\n", color("reset");
		&datab::warr_message("MESSAGE: In database is currently storage $n_err of records.\n");
		
		#B UPDATE - SET status
		if ($params{'mode'} eq 'update')
		{
			$db->do("UPDATE 1_Sequence SET 1_Sequence.1_status = 0;");
			$db->do("UPDATE 1_Sequence,6_GO_parse SET 1_Sequence.1_status = 2 WHERE 1_Sequence.1_seq_name LIKE 6_GO_parse.6_seq_name;");
		}

		#C Get number of errors
		$n_err = 0;
		$n_err = $db->do("SELECT 1_seq_name FROM 1_Sequence WHERE 1_status < 2");
		if ($n_err > 0)
		{
			print STDERR color("blue"), "MESSAGE: In database can be reanotated $n_err records.\n", color("reset");
			&datab::warr_message("MESSAGE: In database can be reanotated $n_err records.\n");	
		}
	
		#D REPAIR/UPDATE sequences in DB
		while (($n_err > 0) && ($repair_mode < 5)) 
		{
			#a Increase repair mode status
			$repair_mode++;
			#b Print status for errors
			print STDERR "MESSAGE: In program was finding $n_err sequences for reanotate in repair cycle $repair_mode.\n";
			&datab::warr_message("MESSAGE: In program was finding $n_err sequences for reanotate in repair cycle $repair_mode.\n");
			#c Get repair cycle - REPAIR FUNCTION
			&update_repair_func($db);
			#d Set sequence status for status 1
			$db->do('UPDATE 1_Sequence SET 1_status = 0 WHERE 1_status = 1');
			#e Detection updateable sequences
			$n_err = 0;
			$n_err = $db->do("SELECT 1_seq_name FROM 1_Sequence WHERE 1_status < 2");
		}
	}
	#4B UPDATE WITH FILE
	else 
	{
		#A Open file with sequences
		if (!(open (ZDROJ, $params{'input'}))) 
		{
			print STDERR color("red"), "ERROR: Input file cannot be open.\n", color("reset");
			&datab::err_message("ERROR: Input file cannot be open.\n");
			$db->disconnect;
			exit (0);
		}

		#B WORK with FILE WITH SEQUENCES
		&work_with_infile(*ZDROJ);
		close(ZDROJ);

		#C REPAIR MODE - get error detection
		$n_err = 0;
		$n_err = $db->do("SELECT 1_seq_name FROM 1_Sequence WHERE 1_status < 2");
		$params{'input'} = '';

		#D If database connection error
		if (!(defined($n_err))) 
		{
			#a Print error
			print STDERR color("red"), "ERROR: Critical error. Trying to reconnect to database system.\n", color("reset");
			&datab::err_message("ERROR: Critical error. Trying to reconnect to database system.\n");
			#b Database reconection
			$db = &datab::pristup($params{'db_host'}, $params{'db_name'}, $params{'db_user'}, $params{'db_heslo'});
			$n_err = 0;
			$n_err = $db->do("SELECT 1_seq_name FROM 1_Sequence WHERE 1_status < 2");
			#c Emergency shut down
			if((!(defined($n_err))) || (!(defined($db)))) 
			{
				print STDERR color("red"), "ERROR: Critical error in database query. Emergency shut down program.\n", color("reset");
				&datab::err_message("ERROR: Critical error in database query. Emergency shut down program.\n");
				exit (0);
			}
		}
		
		#E Print MESSAGE about count of errors
		if ($n_err > 0)
		{
			print STDERR color("blue"), "MESSAGE: In database can be reanotated $n_err records.\n", color("reset");
			&datab::warr_message("MESSAGE: In database can be reanotated $n_err records.\n");	
		}	

		#F REPAIR CYCLE
		while (($n_err > 0) && ($repair_mode < 5)) 
		{
			#a Increase repair mode
			$repair_mode++;
			#b Print message
			print STDERR color("blue"), "MESSAGE: In program was founded $n_err errors for repair in repair cycle $repair_mode.\n", color("reset");
			&datab::warr_message("MESSAGE: In program was founded $n_err errors for repair in repair cycle $repair_mode.\n");
			#c Get repair cycle - REPAIR FUNCTION
			&update_repair_func($db);
			#d Set sequence status for status 1
            		$db->do('UPDATE 1_Sequence SET 1_status = 0 WHERE 1_status = 1');
            		#e Detection updateable sequences
            		$n_err = 0;
            		$n_err = $db->do("SELECT 1_seq_name FROM 1_Sequence WHERE 1_status < 2");
		}		
	}

	#6 MESSAGE about annotationable sequences
	$n_err = $db->do("SELECT 1_seq_name FROM 1_Sequence WHERE 1_status < 2");
	if ($n_err > 0)
	{
		print STDERR color("green"), "MESSAGE: In database can be still repair $n_err of sequences. See the manual for repair mode.\n", color("reset");
		&datab::warr_message("MESSAGE: In database can be still repair $n_err of sequences. See the manual for repair mode.\n");
	}
	else
	{
		print STDERR color("green"), "MESSAGE: All sequences for repair were annotated.\n", color("reset");
		&datab::warr_message("MESSAGE: All sequences for repair were annotated.\n");
	}

	#7 Calculation GO tree
	if (!(&datab::calculating_GOtree($db))) 
	{
		$db->disconnect;
		exit(0);
	}

	#8 Export stat file
	&datab::stat_file($params{'db_host'}, $params{'db_name'}, $params{'db_user'}, $params{'db_heslo'}, 0, 0, 0);

	#9 Mode update succesfully finished - UPDATE
	if ($params{'mode'} eq 'update')
	{
		print STDERR color("green"), "MESSAGE: Program launch is completed in UPDATE mode.\n", color("reset");
		&datab::warr_message("MESSAGE: Program launch is completed in UPDATE mode.\n");
		$db->disconnect;
		exit (1);
	}
	else
	{
		print STDERR color("green"), "MESSAGE: Program launch is completed in REPAIR mode.\n", color("reset");
		&datab::warr_message("MESSAGE: Program launch is completed in REPAIR mode.\n");
		$db->disconnect;
		exit (1);
	}
}

########################################################
#3# = If program mode is analysis
########################################################
if (	defined($params{'input'}) && defined($params{'db_name'}) && 
	defined($params{'db_heslo'}) && defined($params{'db_user'}) && 
	($params{'mode'} eq 'analysis')
)
{
	#1 Database connection
	if (!($db = &datab::pristup($params{'db_host'}, $params{'db_name'}, $params{'db_user'}, $params{'db_heslo'}))) 
	{
		exit(0);
	}
	
	#2 Insert csv or tab delimited file
	if ($params{'input'} =~ /.csv/)
	{
		if (!(&datab::ins_dfg_csv($db, $params{'input'})))
		{
			exit (0);
		}
	}
	else
	{
		if (!(&datab::ins_dfg_txt($db, $params{'input'}))) 
		{
			exit (0);
		}
	}

	#3 Verify GO data
	if(!(&datab::verif_GOparse($db)))
	{
		#A Print warring message
		print STDERR color("red"), "WARRNING: GO data in database are not properly actualized. Trying to recalculate it...\n", color("reset");
		&datab::warr_message("WARRING: GO data in database are not properly actualized. Trying to recalculate it...\n");

		#B Calculation GO tree
		if (!(&datab::calculating_GOtree($db))) 
		{
            		$db->disconnect;
			exit (0);
        	}
	}

	#4 Analysis function
	if (&datab::analysis($db, $params{'pAdj_cut'}, $params{'log_cut'}, $params{'cut_hist'}, $params{'out_pref'}, $params{'out_format'}))
	{
		print STDERR color("green"), "MESSAGE: Analysis for DEGs was successfull.\n", color("reset");
		&datab::warr_message("MESSAGE: Analysis for DEGs was successfull.\n");
	}
	else
	{
		print STDERR color("red"), "ERROR: Analysis faill. Please check the manual.\n", color("reset");
		&datab::err_message("ERROR: Analysis faill. Please check the manual.\n");
		$db->disconnect;
		exit (0);
	}
	
	#5 Export ANNOTATION FILE
	if(&datab::export_annotation_txt($db, $params{'out_pref'} . "_Annotation.txt", 1))
	{
		print STDERR color("green"), "MESSAGE: Export annotation file input DEGs information was successfull.\n", color("reset");
		&datab::warr_message("MESSAGE: Export annotation file input DEGs information was successfull.\n");
		# Create statistics file and get data
        	&datab::stat_file($params{'db_host'}, $params{'db_name'}, $params{'db_user'}, $params{'db_heslo'}, 1, $params{'pAdj_cut'}, $params{'log_cut'});
		$db->disconnect;
		print STDERR color("green"), "MESSAGE: Program launch is completed in ANALYSIS mode.\n", color("reset");
	    &datab::warr_message("MESSAGE: Program launch is completed in ANALYSIS mode.\n");

		exit (1);
	}
	else
	{
		$db->disconnect;
		exit (0);
	}
}

########################################################
#4# = If program mode is delete
########################################################
if (	defined($params{'db_name'}) && defined($params{'db_heslo'}) && 
		defined($params{'db_user'}) && ($params{'mode'} eq 'delete')
)
{
	&datab::delete_dat($params{'db_host'}, $params{'db_user'}, $params{'db_heslo'}, $params{'db_name'});
	exit (1);
}

########################################################
#5# = If program mode is show
########################################################
if (	defined($params{'db_heslo'}) && defined($params{'db_user'}) && 
		($params{'mode'} eq 'show')
)
{
	&datab::show_databases($params{'db_host'}, $params{'db_user'}, $params{'db_heslo'});
	exit (1);
}

########################################################
#6# = If program mode is export
########################################################
if (	defined($params{'db_heslo'}) && defined($params{'db_user'}) && 
		defined($params{'db_name'}) && ($params{'mode'} eq 'export')
)
{
	#1 Database connection
	if (!($db = &datab::pristup($params{'db_host'}, $params{'db_name'}, $params{'db_user'}, $params{'db_heslo'}))) 
	{
		exit(0);
	}
	#2 Export table SEKVENCE
	if (!(&datab::export_table_txt($db, "1_Sequence", "")))
	{
		$db->disconnect;
		exit(0);
	}
	#3 Export table HITS
	if (!(&datab::export_table_txt($db, "2_Hits", "Seq_name \t Hit_definition \t Hit_accession \t Hit_len \t Bit_score \t Score \t Evalue \t Query_from \t Query_to \t Identity \t Positive \t Align_length\n")))
	{
		$db->disconnect;
		exit(0);
	}
	#4 Export table IPR
	if (!(&datab::export_table_txt($db, "3_InterProScan_data", "Seq_name \t Database \t Score \t Evalue \t Signature_name \t Signature_desc \t Signature_ac \t Entry_type \t Entry_name \t Entry_desc \t Entry_ac\n")))
	{
		$db->disconnect;
		exit(0);
	}
	#5 Export table GO_parse
	if (!(&datab::export_table_txt($db, "6_GO_parse", "Seq_name \t Entry_ac \t GO_number\n")))
	{
		$db->disconnect;
		exit(0);
	}
	#6 Export table GO_analysis (all GO numbers for SEQ)
	if (!(&datab::export_table_txt($db, "8_GO_analysis", "Level \t Group \t Seq_name \t Term_id \t GO_number \t Term_name\n")))
	{
		$db->disconnect;
		exit(0);
	}
	#7 Export annotation file for all genes
	if (!(&datab::export_annotation_txt($db, "Annotation_export.txt", 0)))
        {
		$db->disconnect;
		exit(0);
	}
	#8 Create statistics file and get data
	&datab::stat_file($params{'db_host'}, $params{'db_name'}, $params{'db_user'}, $params{'db_heslo'}, 0, 0, 0);
	#9 disconnect DB and exit
	print STDERR color("green"), "MESSAGE: Program launch is completed in EXPORT mode.\n", color("reset");
	&datab::warr_message("MESSAGE: Program launch is completed in EXPORT mode.\n");
	$db->disconnect;
	exit (1);
}

########################################################
#7# = If program mode is import
########################################################
if (	defined($params{'db_heslo'}) && defined($params{'db_user'}) && 
	defined($params{'db_name'}) && defined($params{'input'}) && 
	($params{'mode'} eq 'import')
)
{
	#1 Database connection
	if (!($db = &datab::pristup($params{'db_host'}, $params{'db_name'}, $params{'db_user'}, $params{'db_heslo'}))) 
	{
		exit(0);
	}
	#2 Insert GO numbers
	if (!(&datab::import_GO($db,$params{'input'})))
	{
		print STDERR color("red"), "ERROR: Insert GO numbers was exited with errors.\n", color("reset");
		&datab::err_message("ERROR: Insert GO numbers was exited with errors.\n");
		$db->disconnect;
		exit(0);
	}
	#3 Calculating GO tree
	if (!(&datab::calculating_GOtree($db))) 
	{
		$db->disconnect;
		exit(0);
	}

	#4 Export stat file
	&datab::stat_file($params{'db_host'}, $params{'db_name'}, $params{'db_user'}, $params{'db_heslo'}, 0, 0, 0);

	#5 Mode insert succesfully finished
	print STDERR color("green"), "MESSAGE: Program launch is completed in IMPORT mode.\n", color("reset");
	&datab::warr_message("MESSAGE: Program launch is completed in IMPORT mode.\n");
	$db->disconnect;
	exit (1);
}

###################################################################################################
###################################### MAIN FUNCTIONS #############################################
###################################################################################################

#MAIN 1# - Function for print help message - function()
#Return: 0
sub print_usage 
{
	print STDERR <<EOF
SATRANS:
===================================================================
SATRANS - Tool for anotation and analysis transcriptomic data
===================================================================

Usage:
	main.pl [required params] [other params]

Params:
	--help           :        : Prints the help message and exit.
	--input          : file   : Path to input file.
	--email          : string : user email for IPR scan. Required only for mode insert/update/repair.
	--threads        : int    : Use this many sequences which will be analyzed or CPUs for local BLAST. (default: 2)
	--mode           : string : Mode for work in program (default: analyze)
								analysis
								repair
								update
								delete
								show
								create
								export
								import
	--out            : string : Prefix or path for output files.
	--out_format	     : string : Type of format of output file. Can be csv or txt. (default: txt)
	--e_value        : double : Expect value (E value) cutoff for the saving of BLAST hits. (default: 0.001)
	--align_length   : int    : Sequence length cutoff value for the saving of BLAST hits. (deafult: 20)
	--max_number_hit : int    : Maximum number of BLAST hits per sequence which will be saved. (default: 20)
	--blast_mode     : string : Selected type of blast used by BLAST. Possible values are blastn, blastp, blastx, tblastn, tblastx. (default: blastn)
	--local          :        : Sets up the local BLAST search.
	--log2f_cut      : double : Log2FoldChange cutoff value. (default: 2.0)
	--cut_hist       : double : Cut-off value (>0) for histogram output file. (default: 0.5)
	--pAdj_cut       : double : PAdj cutoff value (from the DESeq2 csv file) (default: 0.01) 
	--db_name        : string : Name of the MySQL database.
	--db_user        : string : Name of the MySQL database user.
    --db_blast       : string : Name of the selected BLAST database (default: nt)
	--db_host        : string : Name of tha host. (default: localhost) 
EOF
}

#MAIN 2# - Function for create searching process and threads - function(sekvence, nazev)
#Return: 0 unknow/error input, 1 correct input, 2 if update  not updateable record
sub find_creat 
{
	#1 Get local variables
	my $sekvence = shift;		# Obsah vkladane sekvence
	my $nazev = shift;			# Nazev sekvnece
	my $verify = 0;				# Overovani druhu sekvence
	my $work_thread = shift;	# ID work thread
	my $db_q = 0;				# Dotaz pro databazi pro DO
	my $order;					# Dotaz pro databazi data

	#2 Verify sequence {0 = ERROR; 1 = NK; 2 = PROT;}
	$verify = &ipr_scan::verify_sequence($sekvence);

	#3 Repair parameters if required
	if ($params{'local'}) 
	{
		$params{'sequence'} = $sekvence;
	}
	else 
	{
		$params{'sequence'} = $nazev . "\n" . $sekvence;
	}
	
	#4 Modify parameter SEQ_ID
	$params{'seq_id'} = $nazev;
	$params{'seq_id'} =~ s/>//g;	# Remove > from sequence
	$params{'seq_id'} =~ s/\n//g;	# Remove /n from name of sequence
	$params{'seq_id'} =~ s/\s//g;

	#5 RESULT FROM VERIFY SEQUENCE
	if (!($verify))
	{
		#A Print ERROR MESSAGE
		print STDERR color("red"), "ERROR: Problem with verify sequence ", $params{'seq_id'}, " . Sequence is probably bad.\n", color("reset");
		&datab::err_message("ERROR: Problem with verify sequence " . $params{'seq_id'} . " . Sequence is probably bad.\n");
		#B Give threads as WAITING
		$vlakna_b[$work_thread] = threads->new(\&cekani, 3, "");
		$vlakna_ipr[$work_thread] = threads->new(\&cekani, 3, "");
		#C SET THREAD CONTROL and return
		$thread_control = 1;
		return 15;
	}	

	#6 Database control
	if (!(defined($db))) 
	{
		#A Reconnect to DB
		$db = &datab::pristup($params{'db_host'}, $params{'db_name'}, $params{'db_user'}, $params{'db_heslo'});
		#B Print ERROR MESSAGE if ERR
		if (!(defined($db)))
		{
			print STDERR color("red"), "ERROR: Problem with database system. Program will be terminated.\n", color("reset");
			&datab::err_message("ERROR: Problem with database system. Program will be terminated.\n");
			exit(0);
		}
	}

	#7 IF SEQ IS ALREADY STORED IN DB - if not UPDATE mode
	$db_q = $db->do('SELECT 1_seq_name FROM 1_Sequence WHERE 1_seq_name LIKE "' . $params{'seq_id'} . '";');
	if (($db_q > 0) && (!($params{'input'} eq ''))) 
	{
		#A Print MESSAGE
		print STDERR color("blue"), "MESSAGE: Sequence with name " . $params{'seq_id'} . " is alredy stored in database.\n", color("reset");
		&datab::warr_message("MESSAGE: Sequence with name " . $params{'seq_id'} . " is alredy stored in database.\n");
		#B Give threads as WAITING
		$vlakna_b[$work_thread] = threads->new(\&cekani, 2, $params{'seq_id'});
		$vlakna_ipr[$work_thread] = threads->new(\&cekani, 2, $params{'seq_id'});
		#C Set thread control and return
		$thread_control = 1;
		return 11;
	}

	#8 INSERT SEQ INTO DATABASE 
	$order = $db->prepare("INSERT INTO 1_Sequence (1_seq_name,1_length,1_sequence,1_status) VALUES (?,?,?,?) ON DUPLICATE KEY UPDATE 1_status = 0;");
	$sekvence =~ s/\s//g;


	#9A SEND SEQ TO DB - status = 0 (currently run)
	if ($order->execute($params{'seq_id'},length($sekvence),$sekvence,0)) 
	{
		#A If repair mode
		if ($repair_mode) 
		{
			print STDERR "Searching for sequence name ", $params{'seq_id'}, " was reinitialize.\n";
		}
		#B If create/updade mode
		else 
		{
			print STDERR "Searching for sequence name ", $params{'seq_id'}, " was initialize.\n";
		}
	}
	#9B If ERROR in SEND SEQ
	else 
	{
		#A Print ERROR MESSAGE
		print STDERR color("red"), "ERROR: Sequence with name ", $params{'seq_id'}, " unable insert into database.\n", color("reset");
		&datab::err_message("ERROR: Sequence with name " . $params{'seq_id'} . " unable insert into database.\n");
		#B Give thread as waiting
		$vlakna_b[$work_thread] = threads->new(\&cekani, 1, "");
		$vlakna_ipr[$work_thread] = threads->new(\&cekani, 1, "");
		#C Set thread control and return
		$thread_control = 1;
		return 0;
	}

	#10 SEARCHING PROCES
	#A - IF NUCLEIC ACID
	if($verify == 1) 
	{
		#a Run BLAST
		$vlakna_b[$work_thread] = threads->new(\&blast::blasting, %params);
		#b Translation into protein
		$params{'sequence'} = $nazev . "\n" . &ipr_scan::prot_best($sekvence);
		#c Run IPR_SCAN
		$vlakna_ipr[$work_thread] = threads->new(\&ipr_scan::send_job_IPR, %params);
		
	}
	#B - IF PROTEIN
	elsif($verify == 2) 
	{
		#a Run BLAST
		$vlakna_b[$work_thread] = threads->new(\&blast::blasting, %params);
		#b Modify SEQ and run IPR_SCAN
		$params{'sequence'} = $nazev . "\n" . $sekvence;
		$vlakna_ipr[$work_thread] = threads->new(\&ipr_scan::send_job_IPR, %params);
	}
	
	#11 Set THREAD_CONTROL and return
	$thread_control = 1;
	return 0;
}

#MAIN 3# - Function for waiting on thread - function(timer)
#Return: 0 pokud uspech
sub cekani 
{
	#1 Get timer
	my $temp = shift;
	my $nazev = shift;
	#2 Wait and return or return
	if ($temp == 1) 
	{
		sleep 2;
		return undef;
	}
	elsif ($temp == 2)
	{
		return $nazev;
	}
	elsif ($temp == 3)
	{
		sleep 2;
		return undef;
	}
	else
	{
		return undef;
	}
}


#MAIN 4# - Function for create and check database for work - function()
#Return: 0 if problem; 1 if success
sub creat_check_db 
{
	#1 Local variables
	my $db;		# Database
	
	#2 Create database
	if (&datab::create_dat($params{'db_host'}, $params{'db_user'}, $params{'db_heslo'}, $params{'db_name'})) 
	{
		print STDERR color("green"), "MESSAGE: Database ", $params{'db_name'}, " succesfully created.\n", color("reset");
		&datab::warr_message("MESSAGE: Database " . $params{'db_name'} . " succesfully created.\n");
	}
	else 
	{
		print STDERR color("red"), "ERROR: Problem in creating database " . $params{'db_name'} . " Please check the manual and your user privilegies.\n", color("reset");
		&datab::err_message("ERROR: Problem in creating database " . $params{'db_name'} . " Please check the manual and your user privilegies.\n");
		return 0;
	}

	#3 Create database approach
	$db = undef;
	$db = &datab::pristup($params{'db_host'}, $params{'db_name'}, $params{'db_user'}, $params{'db_heslo'});
	if (!(defined($db))) 
	{
		return 0;
	}
	
	#4 Verify database structure
	if (!(&datab::verify_datab_structure($db,0))) 
	{
		$db->disconnect;
		return 0;
	}
	
	#5 If success disconnest and return
	$db->disconnect;
	return 1;
}


#MAIN 5# - Function for create and direction of threads for INPUT FILE - function(FILE*)
#Return: 0 if problem; 1 if success
sub work_with_infile 
{
	#1 Get open file
	local(*DATA) = @_;

	#2 Local variables - data
	my $nazev;				# Name of sequence includes >
	my $cteni;				# Active row
	my $ret_blast = undef;	# Ret value from BLAST thread
	my $ret_ipr = undef;	# Ret value from IPR thread
	my $db_act;				# Database action
	my $sekvence;			# Query sequence
	my $promena;			# Variable for waiting on thread
	my $temp;				# Temporalz variable for name of sequence

	#3 Local variables - numeric
	my $read_status = 0;	# Status for read of sequence - 0 empty file; 1 foundedn >; 2 sequence readed
	my $n_aktiv_vlaken = 0;	# Number of active threads
	my $n_total = 0;		# Count of threads
	my $local_c = 0;		# Local counter of sequences
	my $local_e = 0;		# Local counter of error for 60ERR
	my $local_ed = 0;		# Local counter of error for 5ERR

	#4 Local variables - iterative
	my $i = 1;		# Counter

	#5 Set Global counter of sequences
	$id = 0;
	
	#6 Read file to first symbol >
	while ($nazev = <DATA>) 
	{
		if ($nazev =~ m/>/) 
		{
			$read_status = 1;
			last;
		}
	}
	
	#7 Control if file is empty (read_status = 0)
	if ($read_status == 0) 
	{
		print STDERR color("red"), "ERROR: Input file is probably empty.\n", color("reset");
		&datab::err_message("ERROR: Input file is probably empty.\n");
		return 0;
	}
	
	#8 Control DB connection
	if (!(defined($db))) 
	{
		print STDERR color("red"), "ERROR: Database is not avaible.\n", color("reset");
		&datab::err_message("ERROR: Database is not avaible.\n");
		return 0;
	}

	#9 File is not empty (read_status = 1)
	while ($cteni = <DATA>) 
	{
		#A Is GO thread finished
		if ((defined($vlakno_GO)) && ($vlakno_GO->is_joinable())) 
		{
			# Join thread
			$vlakno_GO->join;
			$vlakno_GO = undef;
			# Print MESSAGE
			print STDERR color("blue"), "MESSAGE: GO thread was joined.\n", color("reset");
			&datab::warr_message("MESSAGE: GO thread was joined.\n");
		}

		#B Sekvence read is completed => THREADS
		if ($read_status == 2)
		{
			do 
			{
				# Set ACTIV THREADS
				$n_aktiv_vlaken = 0;
				# Search running threads
				for ($i = 1; $i <= 14; $i++)
				{
					#a Both threads undefined
					if (!((defined($vlakna_b[$i])) && (defined($vlakna_ipr[$i])))) 
					{
						$work_thread = $i;
						last;
					}
					#b One thread is joinable
					elsif((!($vlakna_b[$i]->is_joinable())) || (!($vlakna_ipr[$i]->is_joinable())))
					{
						if (($vlakna_b[$i]->is_running()) || ($vlakna_ipr[$i]->is_running())) 
						{
							$n_aktiv_vlaken++;
						}
					}
					#c Both threads are joinable
					elsif(($vlakna_b[$i]->is_joinable()) && ($vlakna_ipr[$i]->is_joinable())) 
					{
						# FINISH thread
						# Return undef
						$ret_blast = undef;
						$ret_ipr = undef;
						# Join treads
						$ret_blast = $vlakna_b[$i]->join;
						$ret_ipr = $vlakna_ipr[$i]->join;
						# Local counter of SEQ and ERR
						if ($local_c < 100)
						{
							$local_c++;
						}
						else
						{
							$local_c = 0;
							$local_e = 0;
							$local_ed = 0;
						}
						if ($local_c%10 == 0)
						{
							$local_ed = 0;
						}
						
						# Database record actualization
						if ((defined($ret_blast)) && ($ret_blast eq $ret_ipr)) 
						{
							# Database action
							$db_act = $db->prepare('UPDATE 1_Sequence SET 1_status = 2 WHERE 1_seq_name LIKE "' . $ret_blast . '"');
							$db_act->execute;
							# ID for sequence
							$id++;
							# Print message for ID
							print STDERR color("green"), "[$id] Search COMPLETED for sequence ", $ret_blast, color("reset"), "\n";
							&datab::warr_message("[$id] Search COMPLETED for sequence " . $ret_blast . "\n");
						}
						# Error control
						else
						{
							$local_e++;
							$local_ed++;
							# If 60ERR per 100SEQ EXIT program
							if ($local_e > 60)
							{
								exit(0);
							}
							# If ERR sleep 10 seconds
							sleep 10;
							# If 5ERR per 10SEQ sleep 60 seconds
							if ($local_ed > 5)
							{
								sleep 50;
							}
						}
						
						# Reinitialization values for threads
						$vlakna_b[$i] = undef;
                        			$vlakna_ipr[$i] = undef;
						# Select work thead
						$work_thread = $i;
						# STOP cycle
						last;
					}
					# In case of undefined status of threads
					else 
					{
						# Print warning message
						print STDERR color("red"), "ERROR: Undefined status for thread detected.\n", color("reset");
						&datab::err_message("ERROR: Undefined status for thread detected.\n");
						# End running threads
						$vlakna_b[$i]->exit;
						$vlakna_ipr[$i]->exit;
						# Reinitialization value for thread
						$vlakna_b[$i] = undef;
                        			$vlakna_ipr[$i] = undef;
						# Increase count of ERR
						$local_c++;
						$local_e++;
						$local_ed++;
						# If more than 60ERR kill program
						if ($local_e > 60)
						{
							exit(0);
						}
					}
				}# END - for cyklus

				#4bB If number af activ threads is bigger or equal max threads
				if ($n_aktiv_vlaken >= $params{'max_threads'}) # Default value 10
				{
					sleep 5;
				}

				#4bC CREATE NEW THREAD
				elsif (!((defined($vlakna_b[$i])) && (defined($vlakna_ipr[$i])))) 
				{
					# If work thread defined
					if ($thread_control)
					{
						# FUNCTION CREATE
						$thread_control = 0;
						&find_creat($sekvence,$nazev,$work_thread);
						$sekvence = '';
					}
					else
					{
						while ($thread_control == 0)
						{
							sleep 1;
						}
						# FUNCTION CREATE
						$thread_control = 0;
						&find_creat($sekvence,$nazev,$work_thread);
						$sekvence = '';
					}
				}
			} while ($n_aktiv_vlaken >= $params{'max_threads'}); # default 10

			#4c Reinitialization parsing sequence
			$read_status = 1;
			$nazev = $temp;

		} # END - if (read = 2)

		#C Search for >
		if ($cteni =~ m/>/) 
		{
			$read_status++;
			$temp = $cteni;
		}
		#D Reading sequence
		$sekvence = $sekvence . $cteni if ($read_status == 1);
	} # END - while cycle

	#10 Send work index
	$work_thread = $params{'max_threads'} + 1;

	#11 CREATE LAST THREAD
	&find_creat($sekvence,$nazev,$work_thread);
	$sekvence = '';
	$nazev = '';
	$n_total = 0;

	#12 Waiting for finish all threads
	print STDERR color("blue"), "MESSAGE: Last search was launched. Program waiting for finish all threads.\n", color("reset");
	&datab::warr_message("MESSAGE: Last search was launched. Program waiting for finish all threads.\n");

	#13 THREAD CONTROL
	do 
	{
		$n_total = 0;
		foreach $promena (@vlakna_b, @vlakna_ipr)
		{
			# Thread is defined
			if(defined($promena)) 
			{
				#A If promena is joinable
				if($promena->is_joinable()) 
				{
					#a Set ret value
					$ret_blast = undef;

					#b Get ret value
					$ret_blast = ($promena->join);

					#c If return value defined increase status in DATABASE
					if (defined($ret_blast)) 
					{
						$db_act = $db->prepare('UPDATE 1_Sequence SET 1_status = 1_status + 1 WHERE 1_seq_name LIKE "' . $ret_blast . '"');
						$db_act->execute;
				
						# Control if SEQUENCE is TOTALY SEARCHED
						$db_act = $db->prepare('SELECT 1_status FROM 1_Sequence WHERE 1_seq_name LIKE "' . $ret_blast . '"');
						$db_act->execute;
						$temp = undef;
						$temp = ($db_act->fetchrow_array);

						# Print MESSAGE if sequence is searched
						if (defined($temp)) 
						{
							if ($temp == 2)
							{
								$id++;
								print STDERR color("green"), "[$id] Search COMPLETED for sequence ", $ret_blast, color("reset"), "\n";
        			                  		&datab::warr_message("[$id] Search COMPLETED for sequence " . $ret_blast . "\n");
							}
						}
					}
				}
				#B If promena is running
				elsif($promena->is_running())
				{
					$n_total++;
				}
			}
		}
	} while ($n_total != 0);

	#14 Print info about infile
	print STDERR color("blue"), "MESSAGE: Work with INFILE is done.\n", color("reset");
    	&datab::warr_message("MESSAGE: Work with INFILE is done.\n");
	
	#15 Return SUCCESS value
	return 1;
}

#MAIN 6# - Function for UPDATE/REPAIR records in database. The records must have status < 2 - function(database)
#Return: 0 if problem; 1 if success
sub update_repair_func 
{
	#1 Local variables - data
	my $db = shift;
	my $records;					# Records about sequences from DB
	my $name;						# Name of work sequence
	my $obsah;						# Work sequence
	my $ret_thread = undef;			# Return value from BLAST
	my $ret_thread_ipr = undef;		# Return value from IPR
	my $db_act;						# Action for DB
	my $promena;					# Temporal thread

	#2 Local variables - numeric
	my $n_vlaken = 0;	# Number of threads 
	my $n_total;		# Number of threads 2
	my $temp = undef;	# Temp variable
	my $local_c = 0;	# Local counter of sequence
	my $local_e = 0;	# Local counter of error for ERR60
	my $local_ed = 0;	# Local counter of error for ERR5

	#3 Local variables - iterative
	my $i = 0;		# Counter for cycle

	#4 Set GLOBAL counter of sequences and work thread
	$id = 0;
	$work_thread = 1;
	
	#5 Get records SEKVENCE with STATUS < 2
	$records = $db->prepare("SELECT 1_seq_name,1_sequence FROM 1_Sequence WHERE 1_status < 2");
	$records->execute();

	#6 Delete fields for thread
	for ($i = 1; $i <= 14; $i++) 
	{
		$vlakna_b[$i] = undef;
		$vlakna_ipr[$i] = undef;
	}

	#7 WORK with sequences
	while (($name, $obsah) = $records->fetchrow_array) 
	{
		#A Modify name of sequence for sending
		$name = ">" . $name . "\n";
		#B WORK with THREADS
		do 
		{
			$n_vlaken = 0;
			for ($i = 1; $i <= 14; $i++)
			{
				#a Both threads UNDEF
				if (!((defined($vlakna_b[$i])) && (defined($vlakna_ipr[$i])))) 
				{
					$work_thread = $i;	
					last;
				}
				#b One thread is active
				elsif((!($vlakna_b[$i]->is_joinable())) || (!($vlakna_ipr[$i]->is_joinable())))
				{
					if ($vlakna_b[$i]->is_running() || $vlakna_ipr[$i]->is_running())
					{
						$n_vlaken++;
					}
				}
				#c Threads are joinable
				elsif (($vlakna_b[$i]->is_joinable()) && ($vlakna_ipr[$i]->is_joinable()))
				{
					# Set rerunt values
					$ret_thread = undef;
					$ret_thread_ipr = undef;
					# Get return value
					$ret_thread = $vlakna_b[$i]->join;
					$ret_thread_ipr = $vlakna_ipr[$i]->join;
					# Increment counter of sequences
					$id++;
					# Increment local counter of sequences
					if ($local_c < 100)
					{
						$local_c++;
					}
					else
					{
						$local_c = 0;
						$local_e = 0;
						$local_ed = 0;
					}
					if ($local_c%10 == 0)
					{
						$local_ed = 0;
					}
					# If return values the same UPDATE RECORD IN DB
					if ((defined($ret_thread)) && (defined($ret_thread_ipr)) && ($ret_thread eq $ret_thread_ipr)) 
					{
						$db_act = $db->prepare('UPDATE 1_Sequence SET 1_status = 2 WHERE 1_seq_name LIKE "' . $ret_thread . '"');
						$db_act->execute;
						# print MESSAGE
						print STDERR color("green"), "[$id from $n_err] Search COMPLETED for sequence ", $ret_thread, color("reset"), "\n";
						&datab::warr_message("[$id from $n_err] Search COMPLETED for sequence " . $ret_thread . "\n");
					}
					# ERROR in SEARCH
					else
					{
						# Increment counter of errors
						$local_e++;
						$local_ed++;
						# If 60ERR per 100SEQ EXIT program
						if ($local_e > 60)
						{
							print STDERR color("red"), "ERROR: Count of errors is more than 60 per 100 SEQ. Program will be terminated.\n", color("reset");
                			    		&datab::err_message("ERROR: Count of errors is more than 60 per 100 SEQ. Program will be terminated.\n");
							exit(0);
						}
						# If ERR sleep 10 seconds
						sleep 10;
						# If 5ERR per 10SEQ sleep 60 seconds
						if ($local_ed > 5)
						{
							sleep 50;
						}						
					}
					# Set values
					$vlakna_b[$i] = undef;
					$vlakna_ipr[$i] = undef;
					$work_thread = $i;
					last;
				}
				#d Undefined status for thread
				else
				{
					# Print warning message
                    			print STDERR color("red"), "ERROR: Undefined status for thread detected.\n", color("reset");
                    			&datab::err_message("ERROR: Undefined status for thread detected.\n");
                    			# End running threads
                    			$vlakna_b[$i]->exit;
                    			$vlakna_ipr[$i]->exit;
                    			# Reinitialization value for thread
                    			$vlakna_b[$i] = undef;
                    			$vlakna_ipr[$i] = undef;
					# Check if ERR
					$local_e++;
					$local_ed++;
					$local_c++;
					if ($local_e > 60)
					{
						print STDERR color("red"), "ERROR: Count of errors is more than 60 per 100 SEQ. Program will be terminated.\n", color("reset");
                    				&datab::err_message("ERROR: Count of errors is more than 60 per 100 SEQ. Program will be terminated.\n");
						exit(0);
					}
				}
			} # END for LOOP
			# Detection count of threads
			if ($n_vlaken >= $params{'max_threads'}) 
			{
				sleep 5;
			}
			elsif (!((defined($vlakna_b[$work_thread])) && (defined($vlakna_ipr[$work_thread]))))
			{
				# If work thread defined
				if ($thread_control)
				{
					# FUNCTION CREATE
					$thread_control = 0;
					&find_creat($obsah,$name,$work_thread);
					$obsah = '';
					$name = '';
				}
				else
				{
					while ($thread_control == 0)
					{
						sleep 1;
					}
					# FUNCTION CREATE
					$thread_control = 0;
					&find_creat($obsah,$name,$work_thread);
					$obsah = '';
					$name = '';
				}
			}
			else
			{
				$n_vlaken = $params{'max_threads'};
			}
		} while ($n_vlaken >= $params{'max_threads'}); # MAX VLAKEN 10
	}

	#8 Waiting on all threads
	print STDERR color("blue"), "MESSAGE: Search for last sequence was launched.\n", color("reset");
	&datab::warr_message("MESSAGE: Search for last sequence was launched.\n");

	#9 THREAD CONTROL
	do 
	{
		$n_total = 0;
		foreach $promena (@vlakna_b, @vlakna_ipr)
		{
			# Thread is defined
            		if(defined($promena)) 
			{
                		if($promena->is_joinable()) 
				{
                			#A Set ret value
                    			$ret_thread = undef;

                    			#B Get ret value
                    			$ret_thread = ($promena->join);

                    			#C If return value defined increase status in DATABASE
                    			if (defined($ret_thread))
                    			{
						#a Update sequence status
                        			$db_act = $db->prepare('UPDATE 1_Sequence SET 1_status = 1_status + 1 WHERE 1_seq_name LIKE "' . $ret_thread . '"');
                        			$db_act->execute;

                        			#a Control if SEQUENCE is TOTALY SEARCHED
                        			$db_act = $db->prepare('SELECT 1_status FROM 1_Sequence WHERE 1_seq_name LIKE "' . $ret_thread . '"');
                        			$db_act->execute;
                        			$temp = undef;
                        			$temp = ($db_act->fetchrow_array);

						#C Print message
                        			if (defined($temp)) 
						{
                        				if ($temp == 2)
                            				{
                            					$id++;
                                				print STDERR color("green"), "[$id from $n_err] Search COMPLETED for sequence ", $ret_thread, color("reset"), "\n";
                                				&datab::warr_message("[$id from $n_err] Search COMPLETED for sequence " . $ret_thread . "\n");
                            				}
                        			}
                    			}
                		}
				elsif($promena->is_running())
                		{
                			$n_total++;
                		}
            		}
        	}
    	} while ($n_total != 0);
    
	#10 Give ret value  
    	return 1;
}
